package presentation;

/**
 * @author Alberto Iachini
 */

import java.awt.GridBagLayout;
import javax.swing.JFileChooser;
import javax.swing.JPanel;
import javax.swing.JButton;
import java.awt.GridBagConstraints;
import javax.swing.JLabel;
import java.awt.Insets;
import java.io.File;

import javax.swing.JScrollPane;
import javax.swing.filechooser.FileNameExtensionFilter;

import javax.swing.JTable;

import presentation.util.Exporter;
import presentation.util.XmlTableOrchestratorFactory;

import control.Orchestrator;

/**
 * This class is the panel used to show the orchestrator and the simulation
 */

public class OrchestratorTablePanel extends JPanel {


	private static final long serialVersionUID = -3051686144124963060L;
	
	private JButton showOrchestratorButton = null;
	private JButton saveToXmlButton = null;
	private JButton showSimulationButton = null;
	
	private JLabel orchestratorLabel = null;
	private JLabel statusLabel = null;
	
	private JScrollPane orchestratorScrollPane = null;
	private OrchestratorFrame owner = null;
	
	private XmlTableOrchestratorFactory fact;

	private int selected = ORCHESTRATOR;
	
	private static final int ORCHESTRATOR=0;
	private static final int SIMULATION=1;
	
private JTable orchestratorTable = null;

	/**
	 * This is the default constructor
	 */
	public OrchestratorTablePanel(OrchestratorFrame frame) {
		super();
		initialize();
		owner = frame;
	}

	/**
	 * This method initializes this
	 * 
	 */
	private void initialize() {
		GridBagConstraints gridBagConstraints21 = new GridBagConstraints();
		gridBagConstraints21.gridx = 0;
		gridBagConstraints21.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints21.ipadx = 62;
		gridBagConstraints21.gridy = 3;
		GridBagConstraints gridBagConstraints5 = new GridBagConstraints();
		gridBagConstraints5.fill = GridBagConstraints.BOTH;
		gridBagConstraints5.gridy = 1;
		gridBagConstraints5.weightx = 10.0;
		gridBagConstraints5.weighty = 10.0;
		gridBagConstraints5.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints5.gridheight = 4;
		gridBagConstraints5.gridx = 1;
		GridBagConstraints gridBagConstraints3 = new GridBagConstraints();
		gridBagConstraints3.gridx = 1;
		gridBagConstraints3.anchor = GridBagConstraints.WEST;
		gridBagConstraints3.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints3.fill = GridBagConstraints.HORIZONTAL;
		gridBagConstraints3.gridy = 0;
		statusLabel = new JLabel();
		statusLabel.setText("Orchestrator");
		GridBagConstraints gridBagConstraints2 = new GridBagConstraints();
		gridBagConstraints2.gridx = 0;
		gridBagConstraints2.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints2.anchor = GridBagConstraints.WEST;
		gridBagConstraints2.ipadx = 37;
		gridBagConstraints2.gridy = 2;
		GridBagConstraints gridBagConstraints1 = new GridBagConstraints();
		gridBagConstraints1.gridx = 0;
		gridBagConstraints1.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints1.anchor = GridBagConstraints.WEST;
		gridBagConstraints1.ipadx = 0;
		gridBagConstraints1.gridy = 0;
		orchestratorLabel = new JLabel();
		orchestratorLabel.setText("Orchestrator Generator");
		GridBagConstraints gridBagConstraints = new GridBagConstraints();
		gridBagConstraints.gridx = 0;
		gridBagConstraints.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints.anchor = GridBagConstraints.WEST;
		gridBagConstraints.ipadx = 23;
		gridBagConstraints.gridy = 1;
		this.setSize(300, 200);
		this.setLayout(new GridBagLayout());
		this.add(getShowOrchestratorButton(), gridBagConstraints);
		this.add(orchestratorLabel, gridBagConstraints1);
		this.add(getShowSimulationButton(), gridBagConstraints2);
		this.add(statusLabel, gridBagConstraints3);
		this.add(getOrchestratorScrollPane(), gridBagConstraints5);
		this.add(getSaveToXmlButton(), gridBagConstraints21);
	}

	/**
	 * This method initializes showOrchestratorButton	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getShowOrchestratorButton() {
		if (showOrchestratorButton == null) {
			showOrchestratorButton = new JButton();
			showOrchestratorButton.setText("Show Orchestrator");
			showOrchestratorButton.setToolTipText("Show the orchestrator table");
			showOrchestratorButton.addActionListener(new java.awt.event.ActionListener() {
				public void actionPerformed(java.awt.event.ActionEvent e) {
					// show orchestrator table
					selected = ORCHESTRATOR;
					orchestratorTable.setModel(fact.getOrchestratorModel());
					statusLabel.setText("Orchestrator");
				}
			});
		}
		return showOrchestratorButton;
	}

	/**
	 * This method initializes showSimulationButton	
	 * 	
	 * @return javax.swing.JButton
	 */
	private JButton getShowSimulationButton() {
		if (showSimulationButton == null) {
			showSimulationButton = new JButton();
			showSimulationButton.setText("Show Simulation");
			showSimulationButton.setToolTipText("Show the simulation table");
			showSimulationButton.addActionListener(new java.awt.event.ActionListener() {
				public void actionPerformed(java.awt.event.ActionEvent e) {
					// show simulation table
					selected = SIMULATION;
					orchestratorTable.setModel(fact.getSimulationModel());
					statusLabel.setText("Simulation");
				}
			});
		}
		return showSimulationButton;
	}

	/**
	 * This method initializes orchestratorScrollPane	
	 * 	
	 * @return javax.swing.JScrollPane	
	 */
	private JScrollPane getOrchestratorScrollPane() {
		if (orchestratorScrollPane == null) {
			orchestratorScrollPane = new JScrollPane();
			orchestratorScrollPane.setViewportView(getOrchestratorTable());
		}
		return orchestratorScrollPane;
	}

	/**
	 * Open a JFileChooser to save the selected table into xml file
	 * 
	 * @return the selected file
	 */
	private File saveXmlFile()
	{
		JFileChooser fc = new JFileChooser();
		File dir = owner.getLastFolder();
		if(dir!=null && dir.isDirectory())
			fc.setCurrentDirectory(dir);
		FileNameExtensionFilter filter = new FileNameExtensionFilter("XML File", "xml");
		fc.setFileFilter(filter);
		fc.showSaveDialog(this);
		owner.setLastFolder(fc.getCurrentDirectory());
		return fc.getSelectedFile();
	}

	/**
	 * This method initializes saveToXmlButton	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getSaveToXmlButton() {
		if (saveToXmlButton == null) {
			saveToXmlButton = new JButton();
			saveToXmlButton.setText("Save To Xml");
			saveToXmlButton.setToolTipText("Save the selected table in a xml file");
			saveToXmlButton.addActionListener(new java.awt.event.ActionListener() {
				public void actionPerformed(java.awt.event.ActionEvent e) {
					
					// save the xml file and the xsl file
					File file = saveXmlFile();
					if(file != null)
					{
						String dir = file.getParent();
						File xslFile = null;
						String xml = null;
						String xsl = null;
						if(selected == ORCHESTRATOR)
						{
							xml = fact.getXmlOrchestrator();
							xslFile = new File(dir, "orch.xsl");
							xsl = fact.getXslOrchestrator();
						}
						else if(selected == SIMULATION)
						{
							xml = fact.getXmlSimulation();
							xslFile = new File(dir, "sim.xsl");
							xsl = fact.getXslSimulation();
						}
						Exporter.exportToText(file, xml);
						Exporter.exportToText(xslFile, xsl);
					}
				}
			});
		}
		return saveToXmlButton;
	}
	
	/**
	 * Initialize the orchestrator panel with the generated orchestator
	 * 
	 * @param orch the orchestrator
	 */
	public void init(Orchestrator orch)
	{
		fact= new XmlTableOrchestratorFactory(orch);
		selected = ORCHESTRATOR;
		orchestratorTable.setModel(fact.getOrchestratorModel());
	}

	/**
	 * This method initializes orchestratorTable	
	 * 	
	 * @return javax.swing.JTable	
	 */
	private JTable getOrchestratorTable() {
		if (orchestratorTable == null) {
			orchestratorTable = new JTable();
			orchestratorTable.setAutoCreateRowSorter(true);
		}
		return orchestratorTable;
	}

}
